const express = require("express");
const passport = require("passport");
const path = require("path");
const multer = require("multer");
const router = express.Router();

const { uploadProfileImage } = require("../middlewares/upload.middleware");
const {
  checkObituaryOwner,
  checkSuperAdmin,
} = require("../middlewares/user.middleware");

// ✅ Multer config
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const fileType = (file.mimetype.includes("audio") && "audio") || "images";
    cb(null, `public/${fileType}/`);
  },
  filename: function (req, file, cb) {
    cb(
      null,
      file.fieldname + "-" + Date.now() + path.extname(file.originalname)
    );
  },
});
const upload = multer({ storage });

// ✅ Import controllers
const {
  index,
  getAllObituary,
  getAdminObituaries,
  getPinnedObituaries,
  createObituary,
  uploadProfileImageObetrius,
  deleteObituary,
  getObituaryByUserId,
  getObituaryById,
  getByAddress,
  updateObituary,
  pinObituary,
  unpinObituary,
  suspendObituary,
  unsuspendObituary,
  addView,
  getObituaryPlanCounts,
  getObituariesByPlan,
} = require("../controller/obituaryController");

// ✅ Routes
router.get("/all", getAllObituary);
router.get("/admin_obituary", getAdminObituaries);
router.get("/pinned", getPinnedObituaries);

router.post(
  "/createObituary",
  passport.authenticate("jwt", { session: false }),
  upload.fields([
    { name: "ProfileImgUrl", maxCount: 1 },
    { name: "slideImage", maxCount: 1 },
    { name: "images", maxCount: 3 },
    { name: "audio", maxCount: 1 },
  ]),
  createObituary
);

router.post(
  "/upload/profile",
  upload.fields([
    { name: "ProfileImgUrl", maxCount: 1 },
    { name: "slideImage", maxCount: 1 },
  ]),
  uploadProfileImageObetrius
);

router.delete(
  "/deleteObituary/:id",
  passport.authenticate("jwt", { session: false }),
  checkObituaryOwner,
  deleteObituary
);

router.get("/getObituaryByUserId/:userId", getObituaryByUserId);
router.get("/getObituaryById/:id", getObituaryById);
router.get("/get/:address", getByAddress);

router.put(
  "/updateObituary/:id",
  passport.authenticate("jwt", { session: false }),
  checkObituaryOwner,
  upload.fields([
    { name: "ProfileImgUrl", maxCount: 1 },
    { name: "slideImage", maxCount: 1 },
    { name: "images", maxCount: 3 },
    { name: "audio", maxCount: 1 },
  ]),
  updateObituary
);

// ✅ Admin-only actions
router.put(
  "/pin/:id",
  passport.authenticate("jwt", { session: false }),
  checkSuperAdmin,
  pinObituary
);
router.put(
  "/unpin/:id",
  passport.authenticate("jwt", { session: false }),
  checkSuperAdmin,
  unpinObituary
);
router.put(
  "/suspend/:id",
  passport.authenticate("jwt", { session: false }),
  checkSuperAdmin,
  suspendObituary
);
router.put(
  "/unsuspend/:id",
  passport.authenticate("jwt", { session: false }),
  checkSuperAdmin,
  unsuspendObituary
);

// ✅ Plans section
router.get("/plans/counts", getObituaryPlanCounts);
router.get("/plans/:plan", getObituariesByPlan);

// ✅ Views
router.put("/add-view/:id", addView);

// Default index
router.get("/", index);

module.exports = router;
