const { Op } = require("sequelize");
const Memorial = require("../models/Memorial");
const Stories = require("../models/Stories");
const User = require("../models/User");
const CoAdmins = require("../models/CoAdmins");

module.exports = {
async create(req, res) {
  try {
    console.log("Request body: ", req.body);
    console.log("File: ", req.file);

    // Check if a file was uploaded
    if (req.file) {
      // Add the file path to the request body
      req.body.image = req.file.path; // Save the image path
    }

    // Create the story with the provided data, including the image path
    const story = await Stories.create({
      title: req.body.title,
      description: req.body.description,
      memorial_id:req.body.memorial_id,
      user_id: req.body.user_id, // Ensure user_id is passed in the request body
      image: req.body.image || null, // If no image, set it as null
    });

    // Return the created story as a response
    return res.status(201).json({
      message: "Story created successfully",
      story, // Return the created story object
    });
  } catch (err) {
    console.error("Error creating story:", err);

    // Send a generic error message with a 500 status code
    return res.status(500).json({
      message: "An error occurred while creating the story.",
      error: err.message || "Unknown error", // Send more specific error info
    });
  }
}

,

  async getAllStories(req, res) {
    try {
      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      const stories = await Stories.findAndCountAll({
        where: {
          ...(req.query.search && {
            [Op.or]: [
              { title: { [Op.regexp]: req.query.search } },
              { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
              { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
              { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
            ],
          }),
        },
        order: [["createdAt", "DESC"]],
        limit: parseInt(pageSize),
        offset: (parseInt(page) - 1) * parseInt(pageSize),
        include: [
          { model: Memorial, as: "memorial" },
          { model: User, as: "user" },
        ],
      });
      const total = Math.ceil(stories.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        stories: stories.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

  async getStoriesByCoAdmins(req, res) {
    try {
      // Step 1: Get all memorials where the user is a co-admin
      const coAdminMemorials = await CoAdmins.findAll({
        where: { user_id: req.params.userId },
        attributes: ["memorial_id"],
      });
      // Extract memorial IDs from the result
      const memorialIds = coAdminMemorials.map(
        (coAdmin) => coAdmin.memorial_id
      );
      console.log({ memorialIds });

      if (memorialIds.length === 0) {
        return []; // No memorials found for this co-admin
      }

      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      // Step 2: Fetch stories linked to these memorial IDs
      const stories = await Stories.findAndCountAll({
        where: { memorial_id: memorialIds },
        ...(req.query.search && {
          [Op.or]: [
            { title: { [Op.regexp]: req.query.search } },
            { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
            { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
            { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
          ],
        }),
        include: [
          {
            model: Memorial,
            as: "memorial",
          },
        ],
      });
      const total = Math.ceil(stories.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        stories: stories.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

  async getStory(req, res) {
    try {
      const story = await Stories.findByPk(req.params.id, {
        include: [
          { model: Memorial, as: "memorial" },
          { model: User, as: "user" },
        ],
      });
      return res.json(story);
    } catch (err) {
      console.log({ err });
    }
  },

  async getStoriesByMemorial(req, res) {
    try {
      const stories = await Stories.findAll({
        where: { memorial_id: req.params.memorialId },
        order: [["createdAt", "DESC"]],
        include: [{ model: User, as: "user" }],
      });
      if (!stories.length)
        return res
          .status(400)
          .json({ message: "There are no stories with that memorial Id" });
      return res.json(stories);
    } catch (err) {
      console.log({ err });
    }
  },

  async updateStory(req, res) {
    try {
      const story = await Stories.findByPk(req.params.id);
      story.title = req.body.title;
      story.description = req.body.description;
      await story.save();
      return res.json({ message: "Updated story successfully!", story });
    } catch (err) {
      console.log({ err });
    }
  },

  async removeStory(req, res) {
    try {
      const story = await Stories.destroy({ where: { id: req.params.id } });
      return res.json({ message: "Removed story successfully!", story });
    } catch (err) {
      console.log({ err });
    }
  },
};
