const CoAdmins = require("../../models/CoAdmins");
const Memorial = require("../../models/Memorial");
const User = require("../../models/User"); // 👈 import User model

module.exports = {
  // 📌 Get all memorials (admin can see all, normal users only their own)
  async getAllMemorial(req, res) {
    try {
      const page = req.query.page ? parseInt(req.query.page) : 1;
      const pageSize = req.query.pageSize ? parseInt(req.query.pageSize) : 4;

      // 🔎 Get current user from DB (based on req.user injected by auth middleware)
      const user = await User.findByPk(req.user.userid);

      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }

      let whereCondition = {};

      if (user.type !== "memorial_admin" && user.type !== "super_admin") {
        // Normal user → restrict to their own memorials as co-admin
        whereCondition = { user_id: req.user.userid };
      }
      // memorial_admin or super_admin → no restriction (see all)

      const memorials = await CoAdmins.findAndCountAll({
        where: whereCondition,
        order: [["createdAt", "DESC"]],
        limit: pageSize,
        offset: (page - 1) * pageSize,
        include: [{ model: Memorial, as: "memorial" }],
      });

      const total = Math.ceil(memorials.count / pageSize);

      return res.status(200).json({
        filters: { page, nextPage: page + 1, total },
        memorials: memorials.rows,
      });
    } catch (error) {
      console.error("❌ getAllMemorial error:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // 📌 Update profile image for a memorial
  async upDateProfileImageMemorial(req, res) {
    try {
      const { memorialId } = req.body;

      if (!req.file) {
        return res.status(400).json({ error: "No file uploaded" });
      }

      let filePath = req.file.path.replace("public", "api"); // normalize path
      const memorial = await Memorial.update(
        { ProfileImgUrl: filePath },
        { where: { id: memorialId } }
      );

      return res.status(200).json({ message: "Upload successful", memorial });
    } catch (error) {
      console.error("❌ upDateProfileImageMemorial error:", error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // 📌 Update memorial data
  async updateMemorial(req, res) {
    try {
      const { id } = req.params;
      const memorial = await Memorial.update(
        { ...req.body },
        { where: { id } }
      );

      return res
        .status(200)
        .json({ message: "Memorial updated successfully", memorial });
    } catch (error) {
      console.error("❌ updateMemorial error:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },
};
