const { Op } = require("sequelize");
const Sequelize = require("sequelize");
const { Category, Memorial } = require("../models");

module.exports = {
  async create(req, res) {
    console.log("res", req.body);
    try {
      const category = new Category({
        title: req.body.title,
        quote: req.body.quote,
        user_id: req.body.user_id,
      });
      if (req.file) {
        let filePath = req?.file?.path;
        filePath = filePath.replace("public", "api");
        req.file.path = filePath;
        category.image = req.file.path;
      }
      await category.save();
      return res.json({ message: "Created a new category" });
    } catch (err) {
      console.log({ err });
      return res.status(400).json({ message: "Something went wrong" });
    }
  },

  async getAllCategories(req, res) {
    try {
      const categories = await Category.findAll();
      return res.json(categories);
    } catch (err) {
      console.log({ err });
      return res.status(400).json({ message: "Something went wrong" });
    }
  },

  async categoriesFilter(req, res) {
    try {
      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;

      // Buscar categorias com total de memorials
      const categories = await Category.findAll({
        where: {
          ...(req.query.search && {
            title: { [Op.regexp]: req.query.search },
          }),
        },
        attributes: {
          include: [
            [
              Sequelize.literal(`(
              SELECT COUNT(*)
              FROM memorial AS m
              WHERE m.category_id = Category.id
            )`),
              "memorialCount",
            ],
          ],
        },
        order: [["createdAt", "DESC"]],
        limit: parseInt(pageSize),
        offset: (parseInt(page) - 1) * parseInt(pageSize),
      });

      const total = await Category.count({
        where: {
          ...(req.query.search && {
            title: { [Op.regexp]: req.query.search },
          }),
        },
      });

      return res.json({
        filters: {
          page: parseInt(page),
          nextPage: parseInt(page) + 1,
          total: Math.ceil(total / pageSize),
        },
        categories,
      });
    } catch (err) {
      console.log({ err });
      return res.status(400).json({ message: "Something went wrong" });
    }
  },
  async singleCategory(req, res) {
    try {
      const category = await Category.findOne({ where: { id: req.params.id } });
      if (!category)
        return res
          .status(400)
          .json({ message: "Category with that ID doesn't exist!" });
      return res.json(category);
    } catch (err) {
      console.log({ err });
      return res.status(400).json({ message: "Something went wrong" });
    }
  },

  async updateCategory(req, res) {
    try {
      const category = await Category.findOne({ where: { id: req.params.id } });
      category.title = req.body.title;
      category.quote = req.body.quote;
      if (req.file) {
        let filePath = req?.file?.path;
        filePath = filePath.replace("public", "api");
        req.file.path = filePath;
        category.image = req.file.path;
      }
      await category.save();
      return res.json({ message: "Updated the category successfully!" });
    } catch (err) {
      console.log({ err });
      return res.status(400).json({ message: "Something went wrong" });
    }
  },

  async removeCategory(req, res) {
    try {
      const category = await Category.destroy({
        where: { id: req.params.id },
      });
      if (!category)
        return res
          .status(400)
          .json({ message: "Category with that ID doesn't exist!" });
      return res.json({ message: "Category is removed successfully!" });
    } catch (err) {
      console.log({ err });
      return res.status(400).json({ message: "Something went wrong" });
    }
  },
async getMemorialsByCategory(req, res) {
  try {
    const category = await Category.findByPk(req.params.id, {
      include: [{ model: Memorial, as: "memorials" }],
    });

    if (!category) {
      return res.status(404).json({ message: "Category not found" });
    }

    res.json(category.memorials); // returns only memorials
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Server error", error: err.message });
  }
}
};
