// controllers/formSchemaController.js
import FormSchema from "../models/FormSchema.js";

export const saveFormSchema = async (req, res) => {
  try {
    const { title, description, schema } = req.body;

    const form = await FormSchema.create({
      title,
      description,
      schema,
      createdBy: req.user._id, // ✅ Save user ID
    });

    res.status(201).json({ message: "Form schema saved", formId: form._id });
  } catch (err) {
    res.status(500).json({ message: "Error saving schema", error: err.message });
  }
};


// controllers/formSchemaController.js

export const getPendingSchemas = async (req, res) => {
  try {
    if (req.user.role !== "admin") {
      return res.status(403).json({ message: "Access denied: Admins only" });
    }

    const pendingSchemas = await FormSchema.find({ status: "pending" }).populate("createdBy", "name email");
    res.status(200).json(pendingSchemas);
  } catch (err) {
    res.status(500).json({ message: "Error fetching pending schemas", error: err.message });
  }
};


// Approve form
export const approveSchema = async (req, res) => {
  try {
    // Check if user is admin (protect middleware must attach role)
    if (req.user.role !== "admin") {
      return res.status(403).json({ message: "Access denied: Admins only" });
    }

    const { id } = req.params;

    const updated = await FormSchema.findByIdAndUpdate(
      id,
      {
        status: "approved",
        approvedBy: req.user._id,
        approvedAt: new Date(), // Optional: add timestamp
      },
      { new: true }
    ).populate("createdBy", "name email"); // Optional: return creator info

    if (!updated) {
      return res.status(404).json({ message: "Form not found" });
    }

    res.status(200).json({
      message: "✅ Form schema approved successfully",
      form: updated,
    });
  } catch (err) {
    console.error("Error approving form:", err);
    res.status(500).json({
      message: "❌ Error approving form schema",
      error: err.message,
    });
  }
};

// Reject form
export const rejectSchema = async (req, res) => {
  try {
    // ✅ Ensure only admins can reject
    if (req.user.role !== "admin") {
      return res.status(403).json({ message: "Access denied: Admins only" });
    }

    const { id } = req.params;

    const updated = await FormSchema.findByIdAndUpdate(
      id,
      {
        status: "rejected",
        approvedBy: req.user._id,
        rejectedAt: new Date(), // optional timestamp
      },
      { new: true }
    ).populate("createdBy", "name email");

    if (!updated) {
      return res.status(404).json({ message: "Form not found" });
    }

    res.status(200).json({
      message: "❌ Form schema rejected successfully",
      form: updated,
    });
  } catch (err) {
    console.error("Error rejecting form:", err);
    res.status(500).json({
      message: "Error rejecting form schema",
      error: err.message,
    });
  }
};



export const getSingleSchema = async (req, res) => {
  try {
    const { id } = req.params;

    const form = await FormSchema.findById(id).populate("createdBy", "name email");

    if (!form) {
      return res.status(404).json({ message: "Form schema not found" });
    }

    res.status(200).json(form);
  } catch (err) {
    console.error("Error fetching schema:", err.message);
    res.status(500).json({ message: "Error fetching schema", error: err.message });
  }
};


// ✅ Get all approved form schemas
export const getApprovedSchemas = async (req, res) => {
  try {
    const approvedSchemas = await FormSchema.find({ status: "approved" }).populate(
      "createdBy",
      "name email"
    );
    res.status(200).json(approvedSchemas);
  } catch (err) {
    res.status(500).json({
      message: "Error fetching approved schemas",
      error: err.message,
    });
  }
};
