import User from '../models/User.js';
import UserDetails from '../models/UserDetails.js';
import fs from 'fs';
import path from 'path';
import OtpModel from '../models/Otp.js';  
import {sendEmail} from '../utils/sendEmail.js'; 

import bcrypt from 'bcryptjs'; 

export const createClient = async (req, res) => {
    try {
        const {
            name,
            email,
            phone,
            tag,
            isBlocked,
            address1,
            address2,
            zip,
            city,
            country,
            deporte,
            subjective,
            objective,
            assessment,
            plan,
            medicalHistory,
        } = req.body;

        let imagePath = null;
        if (req.file) {
            imagePath = path.join('upload', 'client_images', req.file.filename);

        }
        const randomPassword = Math.random().toString(36).slice(-8); // e.g., 'a2k9jd3l'
        const hashedPassword = await bcrypt.hash(randomPassword, 10);
        // ✅ Create User
        const newUser = await User.create({
            email,
            firstName: name,
            phone,
            password: hashedPassword,
            role: 'patient',
        });

        // ✅ Create User Details
        await UserDetails.create({
            userId: newUser._id,
            address1,
            address2,
            zip,
            city,
            country,
            tag,
            isBlocked,
            deporte,
            subjective,
            objective,
            assessment,
            plan,
            medicalHistory,
            image: imagePath,
        });

        res.status(201).json({
            status: true,
            message: 'Client created successfully',
        });
    } catch (error) {
        console.error(error);
        res.status(500).json({ status: false, message: 'Error creating client' });
    }
};


// GET ALL CLIENTS
export const getAllClients = async (req, res) => {
    try {
        const clients = await User.find({ role: 'patient' }).lean();

        const populatedClients = await Promise.all(clients.map(async (user) => {
            const details = await UserDetails.findOne({ userId: user._id }).lean();
            return { ...user, details };
        }));

        res.json({ status: true, clients: populatedClients });
    } catch (error) {
        console.error(error);
        res.status(500).json({ status: false, message: 'Error fetching clients' });
    }
};

// GET CLIENT BY ID
export const getClientById = async (req, res) => {
    try {
        const { id } = req.params;

        const user = await User.findById(id).lean();
        if (!user) return res.status(404).json({ status: false, message: 'Client not found' });

        const details = await UserDetails.findOne({ userId: id }).lean();

        res.json({ status: true, client: { ...user, details } });
    } catch (error) {
        console.error(error);
        res.status(500).json({ status: false, message: 'Error fetching client' });
    }
};

// UPDATE CLIENT
export const updateClient = async (req, res) => {
    try {
        const { id } = req.params;
        const {
            name, email, phone, tag, isBlocked,
            address1, address2, zip, city, country,
            deporte, subjective, objective, assessment, plan, medicalHistory
        } = req.body;

        const user = await User.findById(id);
        if (!user) return res.status(404).json({ status: false, message: 'Client not found' });

        user.email = email || user.email;
        user.firstName = name || user.firstName;
        user.phone = phone || user.phone;
        await user.save();

        const details = await UserDetails.findOne({ userId: id });
        if (!details) return res.status(404).json({ status: false, message: 'Client details not found' });

        // Handle new image if uploaded
        if (req.file) {
            if (details.image && fs.existsSync(details.image)) {
                fs.unlinkSync(details.image);
            }
            details.image = req.file.path;
        }

        Object.assign(details, {
            address1, address2, zip, city, country,
            tag, isBlocked, deporte, subjective, objective, assessment, plan, medicalHistory
        });

        await details.save();

        res.json({ status: true, message: 'Client updated successfully' });
    } catch (error) {
        console.error(error);
        res.status(500).json({ status: false, message: 'Error updating client' });
    }
};

// DELETE CLIENT
export const deleteClient = async (req, res) => {
    try {
        const { id } = req.params;

        const user = await User.findByIdAndDelete(id);
        const details = await UserDetails.findOneAndDelete({ userId: id });

        if (details?.image && fs.existsSync(details.image)) {
            fs.unlinkSync(details.image);
        }

        if (!user) return res.status(404).json({ status: false, message: 'Client not found' });

        res.json({ status: true, message: 'Client deleted successfully' });
    } catch (error) {
        console.error(error);
        res.status(500).json({ status: false, message: 'Error deleting client' });
    }
};
export const blockClient = async (req, res) => {
    const { id } = req.params;
    const { isBlocked } = req.body;

    try {
        // Find user details by userId (foreign key)
        const details = await UserDetails.findOne({ userId: id });

        if (!details) {
            return res.status(404).json({ message: 'Client details not found.' });
        }

        details.isBlocked = isBlocked;
        await details.save();

        res.json({ message: `Client has been ${isBlocked ? 'blocked' : 'unblocked'}.` });
    } catch (err) {
        console.error('Error blocking client:', err);
        res.status(500).json({ message: 'Server error while blocking client.' });
    }
};


export const deleteMultipleClients = async (req, res) => {
  try {
    const { ids } = req.body; // expecting: { ids: ['id1', 'id2', ...] }

    if (!Array.isArray(ids) || ids.length === 0) {
      return res.status(400).json({ status: false, message: 'No client IDs provided' });
    }

    const users = await User.find({ _id: { $in: ids } });
    const userDetails = await UserDetails.find({ userId: { $in: ids } });

    // Delete images if exist
    for (let details of userDetails) {
      if (details.image && fs.existsSync(details.image)) {
        fs.unlinkSync(details.image);
      }
    }

    await User.deleteMany({ _id: { $in: ids } });
    await UserDetails.deleteMany({ userId: { $in: ids } });

    res.json({ status: true, message: 'Clients deleted successfully' });
  } catch (error) {
    console.error('Error deleting clients:', error);
    res.status(500).json({ status: false, message: 'Failed to delete clients', error });
  }
};


export const blockMultipleClients = async (req, res) => {
  const { ids, isBlocked } = req.body; // expects: { ids: ['id1', 'id2'], isBlocked: true }

  if (!Array.isArray(ids) || typeof isBlocked !== 'boolean') {
    return res.status(400).json({ message: 'Invalid request data.' });
  }

  try {
    const updated = await UserDetails.updateMany(
      { userId: { $in: ids } },
      { $set: { isBlocked } }
    );

    res.json({
      message: `Clients have been ${isBlocked ? 'blocked' : 'unblocked'} successfully.`,
      modifiedCount: updated.modifiedCount
    });
  } catch (err) {
    console.error('Error updating block status:', err);
    res.status(500).json({ message: 'Server error while updating block status.' });
  }
};



export const sendOtp = async (req, res) => {
const { id } = req.body; // this will be an array
console.log("clientBlock",id);



  try {
    const user = await User.findById(id);
    if (!user) return res.status(404).json({ message: "User not found" });

    const email = user.email;

    // Generate 6-digit OTP
    const otpCode = Math.floor(100000 + Math.random() * 900000).toString();

    // Save OTP in DB
    await OtpModel.findOneAndUpdate(
      { email }, // search by email
      {
        email,
        code: otpCode,
        expiresAt: Date.now() + 10 * 60 * 1000 // 10 min expiry
      },
      { upsert: true, new: true }
    );

    const resetLink = `http://localhost:5173/reset-password?email=${encodeURIComponent(email)}`;

const html = `
  <p>Your OTP code is <b>${otpCode}</b>. It will expire in 10 minutes.</p>
  <p>Click here to reset your password: <a href="${resetLink}">${resetLink}</a></p>
`;

  
    await sendEmail(email, 'Password Reset OTP', html);

    res.json({ message: "OTP sent to user's email." });
  } catch (err) {
    console.error("Error sending OTP:", err);
    res.status(500).json({ message: "Server error while sending OTP." });
  }
};


export const resetPasswordController = async (req, res) => {
  const { email, otp, newPassword } = req.body;


  const code = otp;
  console.log("email, code, newPassword ",email, code, newPassword );
  try {
    const existingOtp = await OtpModel.findOne({ email, code });

    if (!existingOtp) return res.status(400).json({ message: "Invalid OTP" });
    if (existingOtp.expiresAt < new Date()) {
      return res.status(400).json({ message: "OTP expired" });
    }

    const hashed = await bcrypt.hash(newPassword, 10);
    await User.findOneAndUpdate({ email }, { password: hashed });

    // Remove OTP after success
    await OtpModel.deleteOne({ _id: existingOtp._id });

    res.json({ message: "Password reset successfully" });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Error resetting password" });
  }
};  