import User from '../models/User.js';
import Category from '../models/Category.js';
import CategoryService from '../models/CategoryService.js'
import Service from '../models/Service.js';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import crypto from 'crypto';
import { sendEmail } from '../utils/sendEmail.js';
import mongoose from 'mongoose';
import Location from '../models/Location.js';
import LocationProvider from '../models/LocationProvider.js';
import ServiceProvider from '../models/ServiceProvider.js';
import ProviderService from '../models/ProviderService.js';
import ProviderLocation from '../models/ProviderLocation.js';
import ServicePageProvider from '../models/ServicePageProvider.js';
import ServiceTime from '../models/ServiceTime.js';
import Client from '../models/Client.js';




// export const registerUser = async (req, res) => {
//   console.log("Request body:", req.body);

//   const {
//     email,
//     role,
//     password,
//     verifyPassword,
//     countryCode,
//     phone,
//     firstName,
//     lastName,
//     sendLoginInfo,
//   } = req.body;

//   try {
//     // Check if email already exists
//     const existingUser = await User.findOne({ email });
//     if (existingUser) {
//       return res.status(400).json({
//         status: false,
//         message: "Email already exists",
//       });
//     }

//     // Validate password match
//     if (password !== verifyPassword) {
//       return res.status(400).json({
//         status: false,
//         message: "Passwords do not match",
//       });
//     }

//     // Hash password
//     const hashedPassword = await bcrypt.hash(password, 10);

//     // Create new user
//     const user = await User.create({
//       email,
//       role: role.toLowerCase(),
//       password: hashedPassword,
//       countryCode,
//       phone,
//       firstName,
//       lastName,
//       sendLoginInfo,
//     });

//     res.status(201).json({
//       status: true,
//       message: "User registered successfully",
//       userId: user._id,
//     });
//   } catch (err) {
//     console.error("Registration Error:", err);
//     res.status(500).json({
//       status: false,
//       message: "Server error",
//       error: err.message,
//     });
//   }
// };


// export const getRegisterUser = async (req, res) => {
//   try {
//     const users = await User.find({ role: { $ne: "admin" } })
//       .select('-password')
//       .sort({ createdAt: -1 }); // ✅ newest users first

//     res.status(200).json(users);
//   } catch (err) {
//     console.error('Error fetching users:', err);
//     res.status(500).json({ message: 'Server error while fetching users' });
//   }
// };

export const loginUser = async (req, res) => {
    const { email, password } = req.body;

    try {
        const user = await User.findOne({ email });
        if (!user || !(await bcrypt.compare(password, user.password))) {
            return res.status(400).json({ status: false, message: 'Invalid credentials' });
        }

        const token = jwt.sign({ id: user._id }, process.env.JWT_SECRET, { expiresIn: '7d' });

        res.json({
            status: true,
            message: 'Login successful',
            token,
            user: { id: user._id, name: user.name, email: user.email, role: user.role }
        });
    } catch (err) {
        res.status(500).json({ status: false, message: 'Server error', error: err.message });
    }
};

export const forgotPassword = async (req, res) => {
    const { email } = req.body;
    try {
        const user = await User.findOne({ email });
        if (!user) return res.status(404).json({ status: false, message: 'User not found' });

        const token = crypto.randomBytes(20).toString('hex');
        user.resetToken = token;
        user.resetTokenExpire = Date.now() + 3600000;
        await user.save();

        const resetLink = `${process.env.FRONTEND_URL}/reset-password/${token}`;
        const html = `
                    <p>You requested a password reset.</p>
                    <p><a href="${resetLink}">Click here to reset your password</a></p>
                    `;

        const result = await sendEmail(user.email, 'Password Reset', html);

        if (result.success) {
            res.json({ status: true, message: 'Reset email sent successfully' });
        } else {
            res.status(500).json({ status: false, message: 'Failed to send email', error: result.error });
        }
    } catch (err) {
        res.status(500).json({ status: false, message: 'Server error', error: err.message });
    }
};

export const resetPassword = async (req, res) => {
    const { token, newPassword } = req.body;

    try {
        const user = await User.findOne({
            resetToken: token,
            resetTokenExpire: { $gt: Date.now() }
        });

        if (!user) return res.status(400).json({ status: false, message: 'Invalid or expired token' });

        user.password = await bcrypt.hash(newPassword, 10);
        user.resetToken = undefined;
        user.resetTokenExpire = undefined;
        await user.save();

        res.json({ status: true, message: 'Password reset successful' });
    } catch (err) {
        res.status(500).json({ status: false, message: 'Server error', error: err.message });
    }
};

export const changePassword = async (req, res) => {
    const { currentPassword, newPassword } = req.body;

    try {
        const user = await User.findById(req.user.id);
        if (!user || !(await bcrypt.compare(currentPassword, user.password))) {
            return res.status(400).json({ status: false, message: 'Invalid current password' });
        }

        user.password = await bcrypt.hash(newPassword, 10);
        await user.save();

        res.json({ status: true, message: 'Password changed successfully' });
    } catch (err) {
        res.status(500).json({ status: false, message: 'Server error', error: err.message });
    }
};

export const editProfile = async (req, res) => {
    const { name } = req.body;

    try {
        const user = await User.findByIdAndUpdate(req.user.id, { name }, { new: true });

        res.json({
            status: true,
            message: 'Profile updated',
            user: { id: user._id, name: user.name, email: user.email }
        });
    } catch (err) {
        res.status(500).json({ status: false, message: 'Server error', error: err.message });
    }
};


// export const deleteUser =async (req,res) => {

//  try {
//     const userId = req.params.id;
// console.log("userid",userId);
// const objectId = new mongoose.Types.ObjectId(userId);


//     const deletedUser = await User.findByIdAndDelete(objectId);
//     console.log("deletedUser",deletedUser);
    

//     if (!deletedUser) {
//       return res.status(404).json({ status: false, message: 'User not found' });
//     }

//     return res.status(200).json({ status: true, message: 'User deleted successfully' });
//   } catch (error) {
//     console.error('Delete user error:', error);
//     return res.status(500).json({ status: false, message: 'Server error' });
//   }
// };



export const EditRegistration = async (req, res) => {
  const userId = req.params.id;
  const updatedData = req.body;

  try {
    const user = await User.findByIdAndUpdate(
      userId,
      updatedData,
      { new: true, runValidators: true }
    );

    if (!user) {
      return res.status(404).json({ status: false, message: 'User not found' });
    }

    res.status(200).json({ status: true, message: 'User updated successfully', user });
  } catch (error) {
    console.error('Update error:', error);
    res.status(500).json({ status: false, message: 'Error updating user', error });
  }
}

// export const Categoryregistration =async (req, res) => {
//    try {
//     const { name, description, position, isVisible, isDefault, image } = req.body;

//     const newCategory = new Category({
//       name,
//       description,
//       position,
//       isVisible: isVisible === 'true',
//       isDefault: isDefault === 'true',
//       image: image || 'No image uploaded',
//     });

//     await newCategory.save();

//     res.status(201).json({ message: 'Category created successfully', category: newCategory });
//   } catch (error) {
//     console.error('Category creation error:', error);
//     res.status(500).json({ message: 'Server error creating category' });
//   }
// };

export const deleteCategory = async (req, res) => {
  try {
    const categoryId = req.params.id;

    // Check if category exists
    const category = await Category.findById(categoryId);
    if (!category) {
      return res.status(404).json({ message: 'Category not found' });
    }

    // Delete the category
    await Category.findByIdAndDelete(categoryId);

    res.status(200).json({ message: 'Category deleted successfully' });
  } catch (error) {
    console.error('Error deleting category:', error);
    res.status(500).json({ message: 'Server error deleting category' });
  }
};



// export const saveCategoryServices = async (req, res) => {
//   try {
//     const { categoryId, services } = req.body;
//     console.log("Payload:", categoryId, services);

//     if (!categoryId || !services || !Array.isArray(services)) {
//       return res.status(400).json({ message: "Category and services are required." });
//     }

//     const newService = new CategoryService({ categoryId, services });
//     await newService.save();

//     res.status(201).json({ message: "Category service saved successfully.", service: newService });
//   } catch (error) {
//     console.error("Error saving category service:", error);
//     res.status(500).json({ message: "Server error", error });
//   }
// };


// export const GetsaveCategoryServices = async (req, res) => {
//   try {
//     const categories = await Category.find();

//     const categoriesWithServices = await Promise.all(
//       categories.map(async (category) => {
//         const categoryServiceDocs = await CategoryService.find({ categoryId: category._id }).populate('services');

//         // flatten all service arrays from multiple documents into a single array
//         const allServices = categoryServiceDocs.flatMap(serviceDoc =>
//           serviceDoc.services.map(service => 
//           ({
//             _id: service._id,
//             name: service.name,
//             price:service.price,
            
//           }))
//         );

//         return {
//           ...category._doc,
//           services: allServices
//         };
//       })
//     );

//     res.json(categoriesWithServices);
//   } catch (err) {
//     console.error("Error fetching categories with services", err);
//     res.status(500).json({ error: "Failed to fetch categories with services" });
//   }
// };




// export const createService = async (req, res) => {
//   try {
//     const { name, description, display, price, tax, image } = req.body;

//     const newCategory = new Service({
//       name,
//       description,
//       display,
//       price,
//       tax,
//       image,
//     });

//     const savedCategory = await newCategory.save();
//     res.status(201).json(savedCategory);
//   } catch (error) {
//     console.error('Error creating category:', error);
//     res.status(500).json({ message: 'Failed to create category' });
//   }
// };




// export const createLocation = async (req, res) => {
//   console.log(req.body);
  
//   try {
//     const {
//       locationName,
//       description,
//       isVisible,
//       isDefault,
//       position,
//       phone,
//       address1,
//       address2,
//       city,
//       zip,
//       country
//     } = req.body;

//     const newLocation = new Location({
//       locationName,
//       description,
//       image: req.file ? `/uploads/${req.file.filename}` : null,
//       isVisible: isVisible === 'true',
//       isDefault: isDefault === 'true',
//       position,
//       address: {
//         phone,
//         address1,
//         address2,
//         city,
//         zip,
//         country,
//       }
//     });

//     await newLocation.save();
//     res.status(201).json({ message: 'Location created successfully!', location: newLocation });
//   } catch (error) {
//     console.error('Error saving location:', error);
//     res.status(500).json({ error: 'Internal server error' });
//   }
// };

// export const getAllLocations = async (req, res) => {
//   try {
//     const locations = await Location.find();

//     const enrichedLocations = await Promise.all(
//       locations.map(async (loc) => {
//         // Get the provider info for this location
//         const locProv = await LocationProvider.findOne({ locationId: loc._id });

//         // const data =  await ProviderService.find().populate('providerId');

//         let providerDetails = [];

//         if (locProv && locProv.providers.length) {
//           providerDetails = await ServiceProvider.find({
//             _id: { $in: locProv.providers }
//           });
//         }

//         return {
//           ...loc._doc,
//           providers: providerDetails
//         };
//       })
//     );

//     res.status(200).json({ status: true, data: enrichedLocations });
//   } catch (error) {
//     console.error('Error fetching locations:', error);
//     res.status(500).json({ status: false, message: 'Internal server error' });
//   }
// };



// export const saveLocationProviders = async (req, res) => {
//   try {
//     const { locationId, providers } = req.body;

//     // Validation
//     if (!locationId || !mongoose.Types.ObjectId.isValid(locationId)) {
//       return res.status(400).json({ error: 'Invalid or missing locationId' });
//     }

//     if (!Array.isArray(providers) || providers.some(p => !mongoose.Types.ObjectId.isValid(p))) {
//       return res.status(400).json({ error: 'Invalid provider IDs' });
//     }

//     // Check if record exists
//     let record = await LocationProvider.findOne({ locationId });

//     if (record) {
//       record.providers = providers;
//       await record.save();
//     } else {
//       record = new LocationProvider({ locationId, providers });
//       await record.save();
//     }

//     res.status(200).json({
//       message: 'Location providers saved successfully',
//       data: record,
//     });
//   } catch (error) {
//     console.error('Error saving providers:', error);
//     res.status(500).json({ error: 'Server error' });
//   }
// };



// export const ServicePage = async (req, res) => {
//   console.log("running");
  
//   try {
//     const { ServicePageId, providers } = req.body;
// console.log("running",providers);
//     // Check if already exists (update if needed)
//     let record = await ServicePageProvider.findOne({ ServicePageId });

//     if (record) {
//       record.providers = providers;
//       await record.save();
//     } else {
//       record = new ServicePageProvider({ ServicePageId, providers });
//       await record.save();
//     }

//     res.status(200).json({ message: 'Location providers saved successfully', data: record });
//   } catch (error) {
//     console.error('Error saving providers:', error);
//     res.status(500).json({ error: 'Server error' });
//   }
// };











// export const saveServiceProvider = async (req, res) => {
//   try {
//     const {
//       name,
//       description,
//       phone,
//       email,
//       clientsAtOnce,
//       showOnBooking,
//     } = req.body;

//     if (!name || !description) {
//       return res.status(400).json({ message: 'Name and description are required' });
//     }

//     const provider = new ServiceProvider({
//       name,
//       description,
//       phone,
//       email,
//       clientsAtOnce,
//       showOnBooking,
//     });

//     await provider.save();

//     res.status(200).json({ message: 'Provider saved successfully', provider });
//   } catch (error) {
//     console.error('Error saving provider:', error);
//     res.status(500).json({ error: 'Server error' });
//   }
// };

// export const getServiceProviders = async (req, res) => {
//   try {
    
//     const services = await ProviderService.find().populate('providerId');

//     // const providers = await ServiceProvider.find();
//     // const id=providers._id;
//     // const existing = await ProviderService.findOne({ id });
//     console.log("providers",services);
    
     
   
//     res.status(200).json({ data: services });
//   } catch (error) {
//     console.error('Error fetching providers:', error);
//     res.status(500).json({ error: 'Server error' });
//   }
// };



// export const saveProviderServices = async (req, res) => {
//   const { providerId, services } = req.body;

//   if (!providerId || !Array.isArray(services)) {
//     return res.status(400).json({ message: 'providerId and services are required.' });
//   }

//   try {
//     // Check if a record already exists for this provider
//     const existing = await ProviderService.findOne({ providerId });

//     if (existing) {
//       // Update existing record
//       existing.services = services;
//       await existing.save();
//       return res.status(200).json({ message: 'Services updated successfully.', data: existing });
//     }

//     // Create new record
//     const newRecord = await ProviderService.create({ providerId, services });
//     return res.status(201).json({ message: 'Services saved successfully.', data: newRecord });

//   } catch (error) {
//     console.error('Error saving provider services:', error);
//     res.status(500).json({ message: 'Internal server error.' });
//   }
// };



export const saveProviderLocations = async (req, res) => {
  const { providerId, locationIds } = req.body;

  if (!providerId || !Array.isArray(locationIds)) {
    return res.status(400).json({ error: 'providerId and locationIds are required.' });
  }

  try {
    // Upsert: if provider exists, update their locations; otherwise, create new entry
    const updated = await ProviderLocation.findOneAndUpdate(
      { providerId },
      { $set: { locationIds } },
      { new: true, upsert: true }
    );

    res.status(200).json({ message: 'Locations saved successfully.', data: updated });
  } catch (err) {
    console.error('Error saving provider locations:', err);
    res.status(500).json({ error: 'Internal server error.' });
  }
};


// export const createServiceTime = async (req, res) => {
//   try {
//     const serviceTime = new ServiceTime(req.body);
//     const saved = await serviceTime.save();
//     res.status(201).json({ status: true, message: "Saved successfully", data: saved });
//   } catch (error) {
//     console.error("Error saving service time:", error);
//     res.status(500).json({ status: false, message: "Failed to save", error });
//   }
// };



// export const updateCategory = async (req, res) => {
//   try {
//     const categoryId = req.params.id;

//     const { name, description, position, isVisible, isDefault, image } = req.body;

//     const updatedCategory = await Category.findByIdAndUpdate(
//       categoryId,
//       {
//         name,
//         description,
//         position,
//         isVisible: isVisible === 'true' || isVisible === true,
//         isDefault: isDefault === 'true' || isDefault === true,
//         image: image || 'No image uploaded',
//       },
//       { new: true }
//     );

//     if (!updatedCategory) {
//       return res.status(404).json({ message: 'Category not found' });
//     }

//     res.status(200).json({ message: 'Category updated successfully', category: updatedCategory });
//   } catch (error) {
//     console.error('Category update error:', error);
//     res.status(500).json({ message: 'Server error updating category' });
//   }
// };





// export const updateCategoryServices = async (req, res) => {
//   try {
//     const categoryId = req.params.id;
//     const { services } = req.body;

//     if (!categoryId || !services || !Array.isArray(services)) {
//       return res.status(400).json({ message: "Category ID and services array are required." });
//     }

//     const serviceIds = services.map(s => {
//       const id = typeof s === 'string' ? s : s._id;
//       if (!mongoose.Types.ObjectId.isValid(id)) {
//         throw new Error(`Invalid service ID: ${id}`);
//       }
//       return new mongoose.Types.ObjectId(id);
//     });

//     const updated = await CategoryService.findOneAndUpdate(
//       { categoryId: new mongoose.Types.ObjectId(categoryId) },
//       { services: serviceIds },
//       { new: true }
//     );

//     if (!updated) {
//       const newService = new CategoryService({
//         categoryId: new mongoose.Types.ObjectId(categoryId),
//         services: serviceIds
//       });

//       await newService.save();
//       return res.status(201).json({ message: 'Service added successfully', service: newService });
//     }

//     res.status(200).json({ message: "Category service updated successfully.", service: updated });
//   } catch (error) {
//     console.error("Error updating category service:", error.message);
//     res.status(500).json({ message: "Server error", error: error.message });
//   }
// };




// export const getService = async (req, res) => {
//   try {
//     const services = await Service.find();
//     res.status(200).json(services);
//   } catch (error) {
//     console.error('Error fetching services:', error);
//     res.status(500).json({ message: 'Failed to fetch services' });
//   }
// };



export const updateLocation = async (req, res) => {
  try {
    const locationId = req.params.id;
    const {
      locationName,
      description,
      isVisible,
      isDefault,
      position,
      phone,
      address1,
      address2,
      city,
      zip,
      country
    } = req.body;

    // Find the location
    const location = await Location.findById(locationId);
    if (!location) {
      return res.status(404).json({ error: 'Location not found' });
    }

    // Update fields
    location.locationName = locationName;
    location.description = description;
    location.isVisible = isVisible === 'true' || isVisible === true;
    location.isDefault = isDefault === 'true' || isDefault === true;
    location.position = position;

    // Update image if a new one is uploaded
    if (req.file) {
      location.image = `/uploads/${req.file.filename}`;
    }

    // Update address
    location.address = {
      phone,
      address1,
      address2,
      city,
      zip,
      country
    };

    await location.save();

    res.status(200).json({ message: 'Location updated successfully!', location });

  } catch (error) {
    console.error('Error updating location:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};



export const getServiceById = async (req, res) => {
  try {
    const service = await Service.findById(req.params.id);
    if (!service) {
      return res.status(404).json({ message: 'Service not found' });
    }

    const serviceTime = await ServiceTime.findOne({ categoryId: req.params.id });

    const servicePageProviders = await ServicePageProvider.find({ ServicePageId: req.params.id })
      .populate('ServicePageId'); // <-- populate Service data if needed

    res.status(200).json({
      service,
      serviceTime,
      servicePageProviders, // ✅ also returning providers
    });
  } catch (error) {
    console.error('Error fetching service by ID:', error);
    res.status(500).json({ message: 'Failed to fetch service' });
  }
};




// export const updateService = async (req, res) => {
//   try {
//     const { id } = req.params;
//     const { name, description, display, price, tax, image } = req.body;

//     const updatedService = await Service.findByIdAndUpdate(
//       id,
//       { name, description, display, price, tax, image },
//       { new: true, runValidators: true }
//     );

//     if (!updatedService) {
//       return res.status(404).json({ message: 'Service not found' });
//     }

//     res.status(200).json(updatedService);
//   } catch (error) {
//     console.error('Error updating service:', error);
//     res.status(500).json({ message: 'Failed to update service' });
//   }
// };



export const deleteLocation = async (req, res) => {
  try {
    const { id } = req.params;
    await Location.findByIdAndDelete(id);
    res.status(200).json({ success: true, message: "Location deleted." });
  } catch (err) {
    res.status(500).json({ success: false, message: "Delete failed", error: err.message });
  }
};




export const createClient = async (req, res) => {
  try {
    const {
      name, email, phone, isBlocked, tag,
      country, address1, address2, zip, city,
      deporte, subjective, objective, assessment, plan, medicalHistory
    } = req.body;

    const image = req.file ? req.file.filename : null;

    const newClient = new Client({
      name,
      email,
      phone,
      isBlocked: isBlocked === 'true',
      tag,
      country,
      address1,
      address2,
      zip,
      city,
      deporte,
      image,
      subjective,
      objective,
      assessment,
      plan,
      medicalHistory
    });

    await newClient.save();
    res.status(201).json({ message: 'Client created successfully', client: newClient });
  } catch (error) {
    console.error('Error creating client:', error);
    res.status(500).json({ message: 'Server error' });
  }
};


export const getClients = async (req, res) => {
  try {
    const clients = await Client.find().sort({ createdAt: -1 }); // latest first
    res.status(200).json(clients);
  } catch (error) {
    console.error('Error fetching clients:', error);
    res.status(500).json({ message: 'Failed to fetch clients' });
  }
};




// export const getServiceProviderById = async (req, res) => {
//   const { id } = req.params;

//   // ✅ Validate ObjectId before querying
//   if (!mongoose.Types.ObjectId.isValid(id)) {
//     return res.status(400).json({ message: 'Invalid service provider ID' });
//   }

//   try {
//     const service = await ServiceProvider.findById(id);
//     if (!service) {
//       return res.status(404).json({ message: 'Service not found' });
//     }

//     const serviceTime = await ProviderService.findOne({ providerId: id });

//     const servicePageProviders = await ProviderLocation.findOne({ providerId: id })
//       .populate('locationIds'); // Optionally populate related data

//     res.status(200).json({
//       service,
//       serviceTime,
//       servicePageProviders,
//     });
//   } catch (error) {
//     console.error('Error fetching service by ID:', error);
//     res.status(500).json({ message: 'Failed to fetch service' });
//   }
// };

export const updateServiceProvider = async (req, res) => {
  try {
    const { id } = req.params;
    const updated = await ServiceProvider.findByIdAndUpdate(id, { $set: req.body }, { new: true });

    if (!updated) return res.status(404).json({ success: false, message: "Provider not found" });

    res.status(200).json({ success: true, data: updated });
  } catch (err) {
    console.error("Update Error:", err);
    res.status(500).json({ success: false, message: "Failed to update provider" });
  }
};


// export const updateProviderServices = async (req, res) => {
//   try {
//     const { id } = req.params;
//     const updated = await ProviderService.findByIdAndUpdate(id, {
//       $set: req.body,
//     }, { new: true });

//     if (!updated) return res.status(404).json({ message: "Not found" });

//     res.status(200).json({ success: true, data: updated });
//   } catch (err) {
//     console.error("Update error:", err);
//     res.status(500).json({ success: false, message: "Update failed" });
//   }
// };



export const updateProviderLocations = async (req, res) => {
  try {
    const { id } = req.params;

    const updated = await ProviderLocation.findByIdAndUpdate(
      id,
      { $set: req.body },
      { new: true }
    );

    if (!updated) return res.status(404).json({ message: "Not found" });

    res.status(200).json({ success: true, data: updated });
  } catch (err) {
    console.error("Update error:", err);
    res.status(500).json({ success: false, message: "Update failed" });
  }
};


// ✅ Block or unblock a client by ID
export const blockClient = async (req, res) => {
  try {
    const clientId = req.params.id;

    const client = await Client.findById(clientId);

    if (!client) {
      return res.status(404).json({ message: 'Client not found' });
    }

    // Toggle or force block
    client.isBlocked = !client.isBlocked;

    await client.save();

    res.status(200).json({
      message: `Client has been ${client.isBlocked ? 'blocked' : 'unblocked'}`,
      isBlocked: client.isBlocked,
    });

  } catch (error) {
    console.error('Error blocking client:', error);
    res.status(500).json({ message: 'Failed to update block status' });
  }
};
