const path = require("path");
const multer = require("multer");

const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const folder = file.mimetype.includes("audio") ? "audio" : "images";
    cb(null, path.join("public", folder));
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(
      null,
      file.fieldname + "-" + uniqueSuffix + path.extname(file.originalname)
    );
  },
});

const upload = multer({ storage });

/**
 * ✅ Bulk upload: images[]
 */
const uploadBulkImages = (fieldName = "images", maxCount = 3) => {
  return upload.array(fieldName, maxCount);
};

/**
 * ✅ Obituary upload: supports slideImage + images[] + ProfileImgUrl
 */
const uploadObituaryImages = upload.fields([
  { name: "ProfileImgUrl", maxCount: 1 }, // optional profile picture
  { name: "slideImage", maxCount: 1 },    // banner
  { name: "images", maxCount: 3 },        // gallery
]);

/**
 * ✅ Map bulk uploaded images (images[])
 */
const mapImagesToFields = (files = []) => {
  const mapped = {};
  if (files[0]) mapped.image1 = `/images/${files[0].filename}`;
  if (files[1]) mapped.image2 = `/images/${files[1].filename}`;
  if (files[2]) mapped.image3 = `/images/${files[2].filename}`;
  return mapped;
};

/**
 * ✅ Map obituary fields into DB-ready format
 */
const mapObituaryImages = (files = {}) => {
  const mapped = {};

  if (files.ProfileImgUrl)
    mapped.ProfileImgUrl = `/images/${files.ProfileImgUrl[0].filename}`;

  if (files.slideImage)
    mapped.SlideImage = `/images/${files.slideImage[0].filename}`;

  if (files.images) {
    if (files.images[0]) mapped.image1 = `/images/${files.images[0].filename}`;
    if (files.images[1]) mapped.image2 = `/images/${files.images[1].filename}`;
    if (files.images[2]) mapped.image3 = `/images/${files.images[2].filename}`;
  }

  return mapped;
};

module.exports = {
  uploadBulkImages,
  mapImagesToFields,
  uploadObituaryImages,
  mapObituaryImages,
};
