const { Op, fn, col } = require("sequelize");
const { Traibute, TributeLike, Memorial, Obituary, User } = require("../models");
const { QueryTypes } = require("sequelize");
const { sequelize } = require("../models"); // adjust path
/**
 * Get all tributes (with optional search)
 */
const getAllTraibute = async (req, res) => {
  try {
    const page = req.query.page ? parseInt(req.query.page, 10) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize, 10) : 4;

    const tributes = await Traibute.findAndCountAll({
      where: {
        ...(req.query.search && {
          [Op.or]: [
            { createdBy: { [Op.regexp]: req.query.search } },
            { "$obituary.Firstname$": { [Op.regexp]: req.query.search } },
            { "$obituary.MiddleName$": { [Op.regexp]: req.query.search } },
            { "$obituary.LastName$": { [Op.regexp]: req.query.search } },
            { "$memorial.Firstname$": { [Op.regexp]: req.query.search } },
            { "$memorial.MiddleName$": { [Op.regexp]: req.query.search } },
            { "$memorial.LastName$": { [Op.regexp]: req.query.search } },
          ],
        }),
      },
      order: [["createdOn", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
      include: [
        { model: Memorial, as: "memorial", required: false },
        { model: Obituary, as: "obituary", required: false },
        { model: User, as: "user" },
      ],
    });

    const total = Math.ceil(tributes.count / pageSize);
    return res.status(200).json({
      filters: { page, nextPage: page + 1, total },
      tributes: tributes.rows,
    });
  } catch (error) {
    console.error("Error fetching all tributes:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

/**
 * Get tributes by user
 */
const getTributesByUser = async (req, res) => {
  try {
    const page = req.query.page ? parseInt(req.query.page, 10) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize, 10) : 4;

    const tributes = await Traibute.findAndCountAll({
      where: {
        userId: req.params.userId,
        ...(req.query.search && {
          [Op.or]: [
            { createdBy: { [Op.regexp]: req.query.search } },
            { "$memorial.Firstname$": { [Op.regexp]: req.query.search } },
            { "$obituary.Firstname$": { [Op.regexp]: req.query.search } },
          ],
        }),
      },
      attributes: {
        include: [[fn("COUNT", col("likes.id")), "likesCount"]],
      },
      include: [
        { model: Memorial, as: "memorial", required: false },
        { model: Obituary, as: "obituary", required: false },
        { model: User, as: "author", attributes: ["userid", "firstName", "lastName", "email"] },
        { model: TributeLike, as: "likes", attributes: [] },
      ],
      group: ["Traibute.id", "memorial.id", "obituary.id", "author.userid"],
      order: [["createdOn", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
      subQuery: false,
    });

    const totalItems = Array.isArray(tributes.count)
      ? tributes.count.length
      : tributes.count;
    const totalPages = Math.ceil(totalItems / pageSize);

    return res.status(200).json({
      filters: { page, nextPage: page + 1, total: totalPages },
      tributes: tributes.rows.map((t) => {
        const tribute = t.toJSON();
        return {
          ...tribute,
          related:
            tribute.postType === "memorial"
              ? tribute.memorial
              : tribute.postType === "obituary"
              ? tribute.obituary
              : null,
        };
      }),
    });
  } catch (error) {
    console.error("Error fetching tributes by user:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

/**
 * Get single tribute
 */
const getSingleTribute = async (req, res) => {
  try {
    const tribute = await Traibute.findOne({
      where: { id: req.params.id },
      include: [
        { model: Memorial, as: "memorial", required: false },
        { model: Obituary, as: "obituary", required: false },
        { model: User, as: "user" },
      ],
    });

    if (!tribute) {
      return res.status(404).json({ error: "Tribute not found" });
    }

    return res.status(200).json({
      message: "Tribute fetched successfully",
      tribute,
    });
  } catch (error) {
    console.error("Error fetching single tribute:", error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

/**
 * Get all tributes for an obituary by id
 */
const getAllObituaryTraibuteById = async (req, res) => {
  const { memorail_id } = req.params;
  try {
    const traibutes = await Traibute.findAll({
      where: { memorailId: memorail_id, postType: "obituary" },
      include: [{ model: User, as: "user" }],
      order: [["createdOn", "DESC"]],
    });
    return res.status(200).json(traibutes);
  } catch (error) {
    console.error("Error fetching obituary tributes:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

/**
 * Get all tributes for a memorial by id
 */
const getAllMemorialTraibuteById = async (req, res) => {
  const { memorail_id } = req.params;

  try {
    const traibutes = await sequelize.query(
      `
      SELECT 
        t.id,
        t.user_id,
        t.memorail_id,
        t.reviewDescription,
        t.traibuteIcon,
        t.postType,
        t.createdOn,
        u.userid AS "user.userid",
        u.firstName AS "user.firstName",
        u.lastName AS "user.lastName",
        u.email AS "user.email"
      FROM Traibutes t
      JOIN Users u 
        ON t.user_id = u.userid
      WHERE t.memorail_id = :memorail_id
        AND t.postType = 'memorial'
      ORDER BY t.createdOn DESC
      `,
      {
        replacements: { memorail_id },
        type: QueryTypes.SELECT,
      }
    );

    return res.status(200).json(traibutes);
  } catch (error) {
    console.error("Error fetching memorial tributes (SQL):", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};



/**
 * Create tribute
 */
const createTraibute = async (req, res) => {
  try {
    const {
      userId,
      isActive,
      createdBy,
      createdOn,
      reviewDescription,
      traibuteIcon,
      memorail_id,
      postType,
    } = req.body;

    const tribute = await Traibute.create({
      userId,
      isActive,
      createdBy,
      createdOn,
      reviewDescription,
      traibuteIcon,
      memorailId: Number(memorail_id),
      postType,
    });

    return res.status(201).json({ message: "Tribute created successfully", tribute });
  } catch (error) {
    console.error("Error creating tribute:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

/**
 * Update tribute
 */
const updateTribute = async (req, res) => {
  try {
    const { createdBy, reviewDescription } = req.body.tribute || req.body;

    const [updated] = await Traibute.update(
      { createdBy, reviewDescription },
      { where: { id: Number(req.params.id) } }
    );
    const tribute = await Traibute.findByPk(req.params.id);
    return res.status(201).json({ message: "Tribute updated successfully", tribute });
  } catch (error) {
    console.error("Error updating tribute:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

/**
 * Delete tribute
 */
const removeTribute = async (req, res) => {
  try {
    const { id } = req.params;
    const tribute = await Traibute.findByPk(id);

    if (!tribute) {
      return res.status(404).json({ error: "Tribute not found" });
    }

    await tribute.destroy();
    return res.status(200).json({ message: "Tribute deleted successfully" });
  } catch (error) {
    console.error("Error deleting tribute:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

module.exports = {
  getAllTraibute,
  getTributesByUser,
  createTraibute,
  removeTribute,
  getAllMemorialTraibuteById,
  getAllObituaryTraibuteById,
  getSingleTribute,
  updateTribute,
};
