const { Op } = require("sequelize");
const MediaContent = require("../models/MediaContent");
const Memorial = require("../models/Memorial");
const MemorialAudioBG = require("../models/MemorialAudioBG");
const ytdl = require("ytdl-core");
const fetch = require("node-fetch");

module.exports = {
  async index(req, res) {
    try {
      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      const media = await MediaContent.findAndCountAll({
        where: {
          ...(req.query.search && {
            [Op.or]: [
              { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
              { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
              { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
            ],
          }),
        },
        limit: parseInt(pageSize),
        offset: (parseInt(page) - 1) * parseInt(pageSize),
        include: [{ model: Memorial, as: "memorial" }],
      });
      const total = Math.ceil(media.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        media: media.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

  async getSingleMedia(req, res) {
    try {
      const media = await MediaContent.findByPk(req.params.id, {
        include: [{ model: Memorial, as: "memorial" }],
      });
      return res.json(media);
    } catch (err) {
      console.log({ err });
    }
  },

  async getAllMediaContent(req, res) {
    try {
      const testimonial = await MediaContent.findAll();
      return res.status(201).json(testimonial);
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async getAllMediaContentOfMemorial(req, res) {
    try {
      const { memorial, fileType } = req.query;
      const data = await MediaContent.findAll({
        where: { memorial_id: memorial, fileType: fileType },
      });
      res.status(201).json(data);
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async createMediaContent(req, res) {
    try {
      const images = await MediaContent.findAll({
        where: { fileType: "images", memorial_id: req.params.memorialId },
      });
      console.log({ images: images.length });
      if (images.length === 100) {
        return res
          .status(400)
          .json({ message: "Exceeded the amount of images by 100" });
      }
      let filePath = req?.file?.path;
      filePath = filePath.replace("public", "api");
      req.file.path = filePath;
      const fileType = req.body.fileType;
      const data = await MediaContent.create({
        content_key: "",
        fileType,
        filePath,
        memorial_id: req.params.memorialId,
      });
      return res
        .status(201)
        .json({ message: "MediaContents created successfully", data });
    } catch (error) {
      console.error(error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async createAudioBackground(req, res) {
    try {
      if (req.file) {
        let filePath = req?.file?.path;

        filePath = filePath.replace("public", "api");
        req.file.path = filePath;
        const data = await MemorialAudioBG.create({
          audio: filePath,
          memorial_id: req.params.memorialId,
        });
        return res
          .status(201)
          .json({ message: "Background audio created successfully", data });
      } else {
        return res.json({ message: "not uploaded" });
      }
    } catch (error) {
      console.error(error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async updateAudioBackground(req, res) {
    try {
      let filePath = req?.file?.path;
      filePath = filePath.replace("public", "api");
      req.file.path = filePath;
      const foundAudio = await MemorialAudioBG.findOne({
        where: { memorial_id: req.params.memorialId },
      });
      if (!foundAudio) {
        const data = await MemorialAudioBG.create({
          audio: filePath,
          memorial_id: req.params.memorialId,
        });
        return res
          .status(201)
          .json({ message: "Background audio created successfully", data });
      }
      const data = await MemorialAudioBG.update(
        {
          audio: filePath,
        },
        { where: { memorial_id: req.params.memorialId } }
      );
      return res
        .status(201)
        .json({ message: "Background audio updated successfully", data });
    } catch (error) {
      console.error(error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async getBGAudioMemorial(req, res) {
    try {
      const audio = await MemorialAudioBG.findOne({
        where: { memorial_id: req.params.memorialId },
      });
      return res.json(audio);
    } catch (error) {
      console.error(error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async createAudioMediaContent(req, res) {
    try {
      const images = await MediaContent.findAll({ fileType: "audio" });
      if (images.length === 100) {
        return res
          .status(400)
          .json({ message: "Exceeded the amount of audio by 100" });
      }
      let filePath = req?.file?.path;
      filePath = filePath.replace("public", "api");
      req.file.path = filePath;
      const fileType = req.body.fileType;
      const data = await MediaContent.create({
        content_key: "",
        fileType,
        filePath,
        memorial_id: req.params.memorialId,
      });
      return res
        .status(201)
        .json({ message: "MediaContents created successfully", data });
    } catch (error) {
      console.error(error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  async createYtContent(req, res) {
    console.log("ssss", { body: req.body });

    try {
      const ytLink = req.body.ytLink;

      // ✅ Improved Regex: Supports youtube.com, youtu.be, vimeo.com + query params
      const isValid = /^(https?:\/\/)?(www\.)?(youtube\.com|youtu\.be|vimeo\.com)\/[^\s]+$/i.test(
        ytLink
      );

      if (!isValid) {
        return res
          .status(400)
          .json({ error: "Only YouTube or Vimeo links allowed" });
      }

      const data = await MediaContent.create({
        content_key: "",
        fileType: "videos",
        filePath: ytLink,
        memorial_id: req.params.memorialId,
        title: req.body.title,
        description: req.body.description
      });

      res
        .status(201)
        .json({ message: "MediaContents created successfully", data });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }

  ,

  async updateMediaImage(req, res) {
    try {
      let filePath = req?.file?.path;
      filePath = filePath.replace("public", "api");
      req.file.path = filePath;
      const media = await MediaContent.findByPk(req.params.id);
      media.filePath = req.file.path;
      await media.save();
      return res.json({ message: "Updated image successfully!", media });
    } catch (err) {
      console.log({ err });
    }
  },

  async updateMediaAudio(req, res) {
    try {
      let filePath = req?.file?.path;
      filePath = filePath.replace("public", "api");
      req.file.path = filePath;
      const media = await MediaContent.findByPk(req.params.id);
      media.filePath = req.file.path;
      await media.save();
      return res.json({ message: "Updated audio successfully!", media });
    } catch (err) {
      console.log({ err });
    }
  },

  async removeMediaContent(req, res) {
    try {
      const { id } = req.params;
      await MediaContent.destroy({ where: { id } });

      return res.status(200).json({ message: "Media deleted successfully" });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },
  async getAllMediaByUser(req, res) {
    try {
      const { userId } = req.params;
      const page = req.query.page ? parseInt(req.query.page, 10) : 1;
      const pageSize = req.query.pageSize ? parseInt(req.query.pageSize, 10) : 4;

      const media = await MediaContent.findAndCountAll({
        include: [
          {
            model: Memorial,
            as: "memorial",
            where: { user_id: userId }, // ✅ only media from memorials owned by this user
          },
        ],
        where: {
          ...(req.query.search && {
            [Op.or]: [
              { "$memorial.Firstname$": { [Op.regexp]: req.query.search } },
              { "$memorial.MiddleName$": { [Op.regexp]: req.query.search } },
              { "$memorial.LastName$": { [Op.regexp]: req.query.search } },
            ],
          }),
        },
        order: [["createdAt", "DESC"]],
        limit: pageSize,
        offset: (page - 1) * pageSize,
      });

      const totalPages = Math.ceil(media.count / pageSize);

      return res.status(200).json({
        filters: { page, nextPage: page + 1, total: totalPages },
        media: media.rows,
      });
    } catch (error) {
      console.error("Error fetching media by user:", error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  }
};
